create or replace package body tbicds.PCK_PATIENT is

  /* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
  /*
  gets basic patient demographic data given the patients fx_user_id
  */
  procedure GetPatientIDRS(pi_vSessionID       in varchar2,
                           pi_vSessionClientIP in varchar2,
                           pi_nUserID          in number,
                           pi_vKey             in varchar2,
                           po_nStatusCode      out number,
                           po_vStatusComment   out varchar2,
                           rs                  out RetRefCursor) 
  is
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    open rs for
      select t.PATIENT_ID,
             tbicds.fnc_utl_decstr(t.FIRST_NAME, pi_vKey, t.PATIENT_ID) as first_name,
             tbicds.fnc_utl_decstr(t.MI, pi_vKey, t.PATIENT_ID) as mi,
             tbicds.fnc_utl_decstr(t.LAST_NAME, pi_vKey, t.PATIENT_ID) as last_name,
             tbicds.fnc_utl_decstr(t.SSN, pi_vKey, t.PATIENT_ID) as ssn,
             mid(tbicds.fnc_utl_decstr(t.last_name, pi_vKey, t.PATIENT_ID),
                 1,
                 1) ||
             mid(tbicds.fnc_utl_decstr(t.ssn, pi_vKey, t.PATIENT_ID), 6, 9) as LNSSNLAST4,
             tbicds.fnc_utl_decstr(t.GENDER, pi_vKey, t.PATIENT_ID) as gender,
             t.MARITAL_STATUS_ID,
             to_date(tbicds.fnc_utl_decstr(t.dob, pi_vKey, t.PATIENT_ID),
                     'MM/DD/YYYY') as dob,
             t.EDUCATION_LEVEL_ID,
             t.PROVIDER_ID,
             t.ADDRESS1,
             t.ADDRESS2,
             t.CITY,
             t.POSTAL_CODE,
             t.HOMEPHONE,
             t.CELLPHONE,
             t.WORKPHONE,
             tbicds.fnc_utl_decstr(t.EMAIL, pi_vKey, t.PATIENT_ID) as email,
             t.STATE_ID,
             t.FX_USER_ID,
             t.LAST_UPDATED,
             t.LAST_UPDATED_BY,
             t.PORTAL_USER_ID,
             t.WRK_PHONE_CALL,
             t.HOME_PHONE_CALL,
             t.HOME_PHONE_MSG,
             t.CELL_PHONE_CALL,
             t.EMAIL_MSG,
             trunc(months_between(SYSDATE,
                                  to_date(tbicds.fnc_utl_decstr(t.dob,
                                                                pi_vKey,
                                                                t.PATIENT_ID),
                                          'MM/DD/YYYY')) / 12) as patient_age
        from tbicds.patient_demographics t
       where t.fx_user_id = pi_nUserID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.GetPatientIDRS(): ' || sqlErrm;
  end;

  /*
  gets dataset to populate Patient Portal Lookup List
  */
  procedure GetPatientPortalListRS(pi_vSessionID       in varchar2,
                                   pi_vSessionClientIP in varchar2,
                                   pi_nUserID          in number,
                                   pi_vKey             in varchar2,
                                   po_nStatusCode      out number,
                                   po_vStatusComment   out varchar2,
                                   rs                  out Pck_Utl_Common.refCursor) 
  is
  begin
  
    --default status to ok
    po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Success;
    po_vStatusComment := null;
  
    open rs for
      select t.patient_id,
             tbicds.fnc_utl_decstr(t.first_name, pi_vKey, t.patient_id) as first_name,
             tbicds.fnc_utl_decstr(t.mi, pi_vKey, t.patient_id) as mi,
             tbicds.fnc_utl_decstr(t.last_name, pi_vKey, t.patient_id) as last_name,
             tbicds.fnc_utl_decstr(t.last_name, pi_vKey, t.patient_id) || ', ' ||
             tbicds.fnc_utl_decstr(t.first_name, pi_vKey, t.patient_id) || ' ' ||
             tbicds.fnc_utl_decstr(t.mi, pi_vKey, t.patient_id) as name,
             t.homephone as phone,
             tbicds.fnc_utl_decstr(t.email, pi_vKey, t.patient_id) as email,
             t.fx_user_id,
             t2.user_name,
             t2.is_locked,
             t2.is_inactive
        from tbicds.patient_demographics t, tbicds.fx_user t2
       where t.fx_user_id = t2.fx_user_id
         and t.provider_id in
             (select provider_id
                from tbicds.app_user s
               where s.dims_id in
                     (select dims_id
                        from tbicds.app_user
                       where fx_user_id = pi_nUserID))
       order by tbicds.fnc_utl_decstr(t.last_name, pi_vKey, t.patient_id),
                tbicds.fnc_utl_decstr(t.first_name, pi_vKey, t.patient_id);
  
  exception
    when others then
    
      po_nStatusCode    := Pck_Utl_Common.c_nResultStatus_Error;
      po_vStatusComment := 'PCK_PATIENT.GetPatientPortalListDS(): ' ||
                           sqlCode || ': ' || sqlErrm;
    
  end;

  /*
   Updates patient demographics data from the demographics page in the application 
   */
  procedure UpdatePatientDemographics(pi_vSessionID       in varchar2,
                                      pi_vSessionClientIP in varchar2,
                                      pi_nUserID          in number,
                                      pi_vKey             in varchar2,
                                      pi_vPatientID       in varchar2,
                                      pi_vFirstName       in varchar2,
                                      pi_vMI              in varchar2,
                                      pi_vLastName        in varchar2,
                                      pi_vSponsorSSN      in varchar2,
                                      pi_vSSN             in varchar2,
                                      pi_vGender          in varchar2,
                                      pi_vDateOfBirth     in varchar2,
                                      pi_vProviderID      in varchar2,
                                      pi_vAddress1        in varchar2,
                                      pi_vAddress2        in varchar2,
                                      pi_vCity            in varchar2,
                                      pi_vPostal_Code     in varchar2,
                                      pi_vHomePhone       in varchar2,
                                      pi_vCellPhone       in varchar2,
                                      pi_vWorkPhone       in varchar2,
                                      pi_vEmail           in varchar2,
                                      pi_vStateID         in varchar2,
                                      pi_nCellPhoneMsg    in number,
                                      pi_nEmailMsg        in number,
                                      pi_nCallPreference  in number,
                                      pi_nPrefLangID      in number,
                                      pi_vPrefLangOther   in varchar2,
                                      pi_nDutyStatusID    in number,
                                      pi_nHandPrefID      in number,
                                      po_nStatusCode      out number,
                                      po_vStatusComment   out varchar2) 
  is
  
    v_nMAJCOM_ID number := 0;
    v_vPMDSQL    long := '';
  
    v_vDOB      varchar2(50) := '';
    v_nCallPref number := null;
  
    v_nHomeCall number := 0;
    v_nCellCall number := 0;
    v_nWorkCall number := 0;
  
    v_nPrefLangID   number := null;
    v_nDutyStatusID number := null;
    v_nHandPrefID   number := null;
  
    v_vProviderID varchar2(50) := null;
  
  begin
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    if pi_nCallPreference > 0 then
      if bitand(pi_nCallPreference, 1) > 0 then
        v_nHomeCall := 1;
      end if;
    
      if bitand(pi_nCallPreference, 2) > 0 then
        v_nCellCall := 1;
      end if;
    
      if bitand(pi_nCallPreference, 4) > 0 then
        v_nWorkCall := 1;
      end if;
    end if;
  
    if pi_nPrefLangID > 0 then
      v_nPrefLangID := pi_nPrefLangID;
    end if;
  
    if pi_nDutyStatusID > 0 then
      v_nDutyStatusID := pi_nDutyStatusID;
    end if;
  
    if pi_nHandPrefID > 0 then
      v_nHandPrefID := pi_nHandPrefID;
    end if;
  
    if pi_vDateOfBirth != '-1' then
      v_vDOB := pi_vDateOfBirth;
    end if;
  
    --get provider id
    begin
      select provider_id
        into v_vProviderID
        from app_user t
       where t.fx_user_id = pi_nUserID;
    exception
      when others then
        v_vProviderID := null;
    end;
  
    update tbicds.patient_demographics
       set first_name      = tbicds.fnc_utl_encstr(upper(pi_vFirstName),
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           mi              = tbicds.fnc_utl_encstr(upper(pi_vMI),
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           last_name       = tbicds.fnc_utl_encstr(upper(pi_vLastName),
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           ssn             = tbicds.fnc_utl_encstr(pi_vSSN,
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           gender          = tbicds.fnc_utl_encstr(pi_vGender,
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           dob             = tbicds.fnc_utl_encstr(v_vDOB,
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           provider_id     = pi_vProviderID,
           address1        = pi_vAddress1,
           address2        = pi_vAddress2,
           city            = upper(pi_vCity),
           postal_code     = pi_vPostal_Code,
           homephone       = pi_vHomePhone,
           cellphone       = pi_vCellPhone,
           workphone       = pi_vWorkPhone,
           email           = tbicds.fnc_utl_encstr(pi_vEmail,
                                                   pi_vKey,
                                                   upper(pi_vPatientID)),
           wrk_phone_call  = pi_nCallPreference,
           home_phone_call = pi_nCallPreference,
           cell_phone_call = pi_nCallPreference,
           home_phone_msg  = pi_nCellPhoneMsg,
           email_msg       = pi_nEmailMsg,
           state_id        = pi_vStateID,
           
           preferred_language_id    = v_nPrefLangID,
           preferred_language_other = pi_vPrefLangOther,
           duty_status_id           = v_nDutyStatusID,
           hand_preference_id       = v_nHandPrefID,
           
           last_updated    = sysdate,
           last_updated_by = pi_nUserID
     where patient_id = pi_vPatientID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.UpdatePatientDemographics(): ' ||
                           sqlErrm;
  end;

  /*get a record set of patients back that match criteria*/
  procedure GetPatientLookupRS(pi_vSessionID       in varchar2,
                               pi_vSessionClientIP in varchar2,
                               pi_nUserID          in number,
                               pi_vKey             in varchar2,
                               pi_nSelectedCases   in number,
                               pi_nSearchType      in number,
                               pi_vSearchValue     in varchar2,
                               po_nStatusCode      out number,
                               po_vStatusComment   out varchar2,
                               rs                  out RetRefCursor) 
  is
  
    v_const_search_last_name number := 1;
    v_const_search_fmp_ssn   number := 3;
   
    strSQL       varchar2(4000);
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    strSQL := 'select ';
    strSQL := strSQL || ' p.PATIENT_ID, ';
  
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.FIRST_NAME, 
              :pi_vKey, p.PATIENT_ID) as first_name, ';
  
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.MI, 
              :pi_vKey, p.PATIENT_ID) as mi, ';
  
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.LAST_NAME, 
              :pi_vKey, p.PATIENT_ID) as last_name, ';
  
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.SSN,
              :pi_vKey, p.PATIENT_ID) as ssn, ';
  
    strSQL := strSQL || ' mid(tbicds.fnc_utl_decstr(p.last_name, 
              :pi_vKey, p.PATIENT_ID), 1, 1) || 
              mid(tbicds.fnc_utl_decstr(p.ssn, 
              :pi_vKey, p.PATIENT_ID), 6, 9) as LNSSNLAST4, ';
  
    strSQL := strSQL || ' mid(tbicds.fnc_utl_decstr(p.last_name, ''' ||
              pi_vKey ||
              ''', p.PATIENT_ID), 1, 1) || mid(tbicds.fnc_utl_decstr(p.ssn, ''' ||
              pi_vKey || ''', p.PATIENT_ID), 6, 9) as fmpssnlast4, ';
  
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.GENDER, ''' || pi_vKey ||
              ''', p.PATIENT_ID) as gender, ';
  
    strSQL := strSQL || ' p.MARITAL_STATUS_ID, ';
  
    strSQL := strSQL || ' to_date(tbicds.fnc_utl_decstr(p.dob, ''' ||
              pi_vKey || ''', p.PATIENT_ID), ''MM/DD/YYYY'') as dob, ';
  
    strSQL := strSQL || ' to_char(to_date(tbicds.fnc_utl_decstr(p.dob, ''' ||
              pi_vKey ||
              ''', p.PATIENT_ID), ''MM/DD/YYYY''), ''MM/DD/YYYY'') as birthdate, ';
  
    strSQL := strSQL || ' p.EDUCATION_LEVEL_ID, ';
    strSQL := strSQL || ' p.PROVIDER_ID, ';
    strSQL := strSQL || ' p.ADDRESS1, ';
    strSQL := strSQL || ' p.ADDRESS2, ';
    strSQL := strSQL || ' p.CITY, ';
    strSQL := strSQL || ' p.POSTAL_CODE, ';
    strSQL := strSQL || ' p.HOMEPHONE, ';
    strSQL := strSQL || ' p.CELLPHONE, ';
    strSQL := strSQL || ' p.WORKPHONE, ';
    
    strSQL := strSQL || ' tbicds.fnc_utl_decstr(p.EMAIL, 
           :pi_vKey, p.PATIENT_ID) as email, ';
    
    strSQL := strSQL || ' p.STATE_ID, ';
    strSQL := strSQL || ' p.FX_USER_ID, ';
    strSQL := strSQL || ' p.LAST_UPDATED, ';
    strSQL := strSQL || ' p.LAST_UPDATED_BY, ';
    strSQL := strSQL || ' p.PORTAL_USER_ID, ';
    strSQL := strSQL || ' p.WRK_PHONE_CALL, ';
    strSQL := strSQL || ' p.HOME_PHONE_CALL, ';
    strSQL := strSQL || ' p.HOME_PHONE_MSG, ';
    strSQL := strSQL || ' p.CELL_PHONE_CALL, ';
    strSQL := strSQL || ' p.EMAIL_MSG, ';
    
    strSQL := strSQL || ' trunc(months_between(SYSDATE, 
           to_date(tbicds.fnc_utl_decstr(p.dob, 
           :pi_vKey, p.PATIENT_ID), ''MM/DD/YYYY'')) / 12) as patient_age, ';
    
    strSQL := strSQL || ' 0 as case_closed ';
    
    strSQL := strSQL || ' from tbicds.patient_demographics p ';
  
    strSQL := strSQL || ' where ';
    
    strSQL := strSQL || ' p.site_id in (';
    
    strSQL := strSQL || 'select site_id ';
    strSQL := strSQL || '  from tbicds.fx_user s ';
    strSQL := strSQL || ' where fx_user_id = :pi_nUserID ) ';
  
    -- is the search by last name?
    if pi_nSearchType = v_const_search_last_name then
  
      strSQL := strSQL || ' and UPPER(';
      
      strSQL := strSQL || 'tbicds.fnc_utl_decstr(p.LAST_NAME, 
                :pi_vKey, p.PATIENT_ID)';
                
      strSQL := strSQL || ') like UPPER(''' || pi_vSearchValue || ''')';
      strSQL := strSQL || ' order by p.last_name, p.first_name asc';
   
    --is the search by lastname initial SS last 4
    elsif pi_nSearchType = v_const_search_fmp_ssn then
  
      strSQL := strSQL || ' and ';
    
      strSQL := strSQL || ' mid(tbicds.fnc_utl_decstr(p.last_name,
                :pi_vKey, p.PATIENT_ID), 1, 1) || 
                mid(tbicds.fnc_utl_decstr(p.ssn, 
                :pi_vKey, p.PATIENT_ID), 6, 9) ';
      strSQL := strSQL || ' = :pi_vSearchValue ';
      strSQL := strSQL || ' order by p.last_name, p.first_name asc';
      
    end if;
  
    if pi_nSearchType = v_const_search_last_name then
  
      open rs for strSQL 
      using pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_nUserID,
            --last name search binds
            pi_vKey;
     
     end if;
    
    if pi_nSearchType = v_const_search_fmp_ssn then
    
      open rs for strSQL 
      using pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_vKey,
            pi_nUserID,
            --LSSN search binds
            pi_vKey,
            pi_vKey,
            pi_vSearchValue;           
    
    end if;
    
         
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.getPatientLookupRS(): ' || sqlErrm;
  end;


  /* Returns the patient demographics recordset  */
  procedure GetPatientDemographicsRS(pi_vSessionID       in varchar2,
                                     pi_vSessionClientIP in varchar2,
                                     pi_nUserID          in number,
                                     pi_vKey             in varchar2,
                                     pi_vPatientID       in varchar2,
                                     po_nStatusCode      out number,
                                     po_vStatusComment   out varchar2,
                                     rs                  out RetRefCursor) 
  is
    --v_nTreatmentID number;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
    --v_nTreatmentID    := 0;
  
   -- select max(treatment_id)
   --   into v_nTreatmentID
   --   from treatment
   --  where patient_id = pi_vPatientID;
  
    open rs for
      select t.PATIENT_ID,
             t.DFN,
             tbicds.fnc_utl_decstr(t.FIRST_NAME, pi_vKey, t.PATIENT_ID) as first_name,
             tbicds.fnc_utl_decstr(t.MI, pi_vKey, t.PATIENT_ID) as mi,
             tbicds.fnc_utl_decstr(t.LAST_NAME, pi_vKey, t.PATIENT_ID) as last_name,
             tbicds.fnc_utl_decstr(t.SSN, pi_vKey, t.PATIENT_ID) as ssn,
             mid(tbicds.fnc_utl_decstr(t.last_name, pi_vKey, t.PATIENT_ID),
                 1,
                 1) ||
             mid(tbicds.fnc_utl_decstr(t.ssn, pi_vKey, t.PATIENT_ID), 6, 9) as LNSSNLAST4,
             tbicds.fnc_utl_decstr(t.GENDER, pi_vKey, t.PATIENT_ID) as gender,
             t.MARITAL_STATUS_ID,
             to_date(tbicds.fnc_utl_decstr(t.dob, pi_vKey, t.PATIENT_ID),
                     'MM/DD/YYYY') as dob,
             t.EDUCATION_LEVEL_ID,
             t.PROVIDER_ID,
             t.ADDRESS1,
             t.ADDRESS2,
             t.CITY,
             t.POSTAL_CODE,
             t.HOMEPHONE,
             t.CELLPHONE,
             t.WORKPHONE,
             tbicds.fnc_utl_decstr(t.EMAIL, pi_vKey, t.PATIENT_ID) as email,
             t.STATE_ID,
             t.FX_USER_ID,
             t.LAST_UPDATED,
             t.LAST_UPDATED_BY,
             t.PORTAL_USER_ID,
             t.WRK_PHONE_CALL,
             t.HOME_PHONE_CALL,
             t.HOME_PHONE_MSG,
             t.CELL_PHONE_CALL,
             t.EMAIL_MSG,
             
             (select s.name
                from tbicds.app_user s
               where s.provider_id = t.PROVIDER_ID) as provider_name,
             trunc(months_between(SYSDATE,
                                  to_date(fnc_utl_decstr(t.dob,
                                                         pi_vKey,
                                                         t.PATIENT_ID),
                                          'MM/DD/YYYY')) / 12) as patient_age,
             '' as MILITARY_STATUS_TITLE,
             '' as MILITARY_SERVICE_TITLE,
             '' as MARITAL_STATUS_TITLE,
             '' as GRADE,
             tbicds.fnc_utl_decstr(t.gender, pi_vKey, pi_vPatientID) as GENDER_DESC,
             '' as BASE,
             0 as referral_count,
             0 as OPENCASE_COUNT,             
             t.wrk_phone_call,
             t.home_phone_call,
             t.home_phone_msg,
             t.email_msg,
             t.preferred_language_id,
             t.preferred_language_other,
             /*tbicds.fnc_utl_decstr(t.preferred_language_other,
                                   pi_vKey,
                                   t.PATIENT_ID) as preferred_language_other,*/
             duty_status_id,
             hand_preference_id
      
        from tbicds.patient_demographics    t
       where t.patient_id = pi_vPatientID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.GetPatientDemographicsRS(): ' ||
                           sqlErrm;
  end;

  /*
  Insert patient demographics
  */
  procedure InsertPatientDemographics(pi_vSessionID       in varchar2,
                                      pi_vSessionClientIP in varchar2,
                                      pi_nUserID          in number,
                                      --Patient Table
                                      pi_vKey            in varchar2,
                                      pi_vPatientID      in varchar2,
                                      pi_vEncounterID    in varchar2,
                                      pi_vFirstName      in varchar2,
                                      pi_vMI             in varchar2,
                                      pi_vLastName       in varchar2,
                                      pi_vSponsorSSN     in varchar2,
                                      pi_vSSN            in varchar2,
                                      pi_vGender         in varchar2,
                                      pi_vDateOfBirth    in varchar2,
                                      pi_vProviderID     in varchar2,
                                      pi_vAddress1       in varchar2,
                                      pi_vAddress2       in varchar2,
                                      pi_vCity           in varchar2,
                                      pi_vPostal_Code    in varchar2,
                                      pi_vHomePhone      in varchar2,
                                      pi_vCellPhone      in varchar2,
                                      pi_vWorkPhone      in varchar2,
                                      pi_vEmail          in varchar2,
                                      pi_vStateID        in varchar2,
                                      pi_nCellPhoneMsg   in number,
                                      pi_nEmailMsg       in number,
                                      pi_nCallPreference in number,
                                      po_nStatusCode     out number,
                                      po_vStatusComment  out varchar2)
  is
  
    v_nRelationshipID number;
    v_nSSNCount       number;
    v_vDOB            varchar2(50) := '';
    v_nCallPref       number := null;
  
    v_nHomeCall number := 0;
    v_nCellCall number := 0;
    v_nWorkCall number := 0;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nRelationshipID := 0;
    v_nSSNCount       := 0;
  
    if pi_nCallPreference > 0 then
      if bitand(pi_nCallPreference, 1) > 0 then
        v_nHomeCall := 1;
      end if;
    
      if bitand(pi_nCallPreference, 2) > 0 then
        v_nCellCall := 1;
      end if;
    
      if bitand(pi_nCallPreference, 4) > 0 then
        v_nWorkCall := 1;
      end if;
    end if;
  
    --SSN Check For Duplicate
    select count(ssn)
      into v_nSSNCount
      from tbicds.patient_demographics t
     where tbicds.fnc_utl_decstr(t.SSN, pi_vKey, t.PATIENT_ID) = pi_vSSN;
  
    if (v_nSSNCount > 0) then
      po_nStatusCode    := Pck_Common.c_nResultStatus_Error;
      po_vStatusComment := po_vStatusComment ||
                           'Social Security Number already exists. ';
    end if;
  
    -- return if error
    if po_nStatusCode = Pck_Common.c_nResultStatus_Error then
      po_vStatusComment := 'Error: ' || rtrim(po_vStatusComment, ', ');
      return;
    end if;
  
    if pi_vDateOfBirth != '-1' then
      v_vDOB := pi_vDateOfBirth;
    end if;
  
    insert into tbicds.patient_demographics t
      (patient_id,
       first_name,
       mi,
       last_name,
       ssn,
       gender,
       dob,
       provider_id,
       address1,
       address2,
       city,
       postal_code,
       homephone,
       cellphone,
       workphone,
       email,
       state_id,
       home_phone_call,
       cell_phone_call,
       wrk_phone_call,
       home_phone_msg,
       email_msg,
       last_updated,
       last_updated_by)
    values
      (upper(pi_vPatientID),
       tbicds.fnc_utl_encstr(upper(pi_vFirstName),
                             pi_vKey,
                             upper(pi_vPatientID)),
       tbicds.fnc_utl_encstr(upper(pi_vMI), pi_vKey, upper(pi_vPatientID)),
       tbicds.fnc_utl_encstr(upper(pi_vLastName),
                             pi_vKey,
                             upper(pi_vPatientID)),
       tbicds.fnc_utl_encstr(pi_vSSN, pi_vKey, upper(pi_vPatientID)),
       tbicds.fnc_utl_encstr(pi_vGender, pi_vKey, upper(pi_vPatientID)),
       tbicds.fnc_utl_encstr(v_vDOB, pi_vKey, upper(pi_vPatientID)),
       pi_vProviderID,
       pi_vAddress1,
       pi_vAddress2,
       upper(pi_vCity),
       pi_vPostal_Code,
       pi_vHomePhone,
       pi_vCellPhone,
       pi_vWorkPhone,
       tbicds.fnc_utl_encstr(pi_vEmail, pi_vKey, upper(pi_vPatientID)),
       pi_vStateID,
       v_nHomeCall,
       v_nCellCall,
       v_nWorkCall,
       pi_nCellPhoneMsg,
       pi_nEmailMsg,
       sysdate,
       pi_nUserID);
    commit;
  
    --Create Initial Patient Sponsor Record Needed By Assessments --Assessments can only to Updates
    /*insert into tbicds.patient_sponsor
      (patient_id, relationship_id, last_updated, last_updated_by)
    values
      (pi_vPatientID, v_nRelationshipID, sysdate, pi_nUserID);
    commit;*/
  
    --Create Initial Emergency Contact Record Needed By Assessments --Assessments can only to Updates
    /*insert into tbicds.patient_emergency_contact
      (patient_id, relationship_id, last_updated, last_updated_by)
    values
      (pi_vPatientID, v_nRelationshipID, sysdate, pi_nUserID);
    commit;*/
  
  --  insert into tbicds.treatment
  --    (treatment_id,
  --     patient_id,
  --     initial_visit_date,
  --     last_updated,
  --     last_updated_by)
  --  values
  --    (1, pi_vPatientID, sysdate, sysdate, pi_nUserID);
  --  commit;
  
    --REVAMP 20130219 DS: As requested by VA (Dr Kuna), do not create a default encounter
    --
    --insert into tbicds.encounter
    --   (encounter_id,
    --    treatment_id,
    --    patient_id,
    --    encounter_date,
    --    last_updated,
    --    last_updated_by)
    -- values
    --   (pi_vEncounterID, 1, pi_vPatientID, sysdate, sysdate, pi_nUserID);
    -- commit;
    --
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.InsertPatientDemographics(): ' ||
                           sqlErrm;
    
  end;

  /*
  does the patient only have 1 encounte for the latest treatment
  */
  procedure InitialEncounter(pi_vSessionID       in varchar2,
                             pi_vSessionClientIP in varchar2,
                             pi_nUserID          in number,
                             pi_vPatientID       in varchar2,
                             po_nInitialEnc      out number,
                             po_nStatusCode      out number,
                             po_vStatusComment   out varchar2) 
  is
  
    v_nCount       number;
    v_nTreatmentID number;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
    v_nCount          := 0;
    v_nTreatmentID    := 0;
    po_nInitialEnc    := 0;
  
    --get the number of encounters for the
    --current open treatment
 --   select count(*)
 --     into v_nCount
 --     from tbicds.encounter e
 --    where e.patient_id = pi_vPatientID
 --      and e.encounter_type_id in (0, 6, 8)
 --      and e.treatment_id =
 --          (select max(treatment_id)
 --             from tbicds.treatment t
 --            where t.patient_id = pi_vPatientID
 --              and nvl(t.case_closed, 0) = 0);
  
    if (v_nCount = 1) then
      po_nInitialEnc := 1;
    end if;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_INTAKE.InitialEncounter(): ' || sqlErrm;
  end;

  /*
  Get patient treatment id recordset
  */
  procedure GetPatientTreatmentIdRS(pi_vSessionID       in varchar2,
                                    pi_vSessionClientIP in varchar2,
                                    pi_nUserID          in number,
                                    pi_vPatientID       in varchar2,
                                    po_nStatusCode      out number,
                                    po_vStatusComment   out varchar2,
                                    rs                  out RetRefCursor)
  is
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
--    open rs for
--      select treatment_id
--        from tbicds.treatment t
--       where patient_id = pi_vPatientID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.GetPatientTreatmentIdRS(): ' ||
                           sqlErrm;
  end;

  /*
  Remove patient intake assessments
  */
  procedure DelIncPatIntakeAssessments(pi_vSessionID       in varchar2,
                                       pi_vSessionClientIP in varchar2,
                                       pi_nUserID          in number,
                                       pi_vPatientID       in varchar2,
                                       po_nStatusCode      out number,
                                       po_vStatusComment   out varchar2)
  is
    v_nNumOfIncAssessments number := 0;
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
   /* for rec in ( -- select all INCOMPLETE encounter intakes for module for patient
                select ei.*
                  from tbicds.encounter_intake ei
                 where ei.encounter_id in
                       ( -- select all encounters for patient
                        select e.encounter_id
                          from tbicds.encounter e
                         where e.patient_id = pi_vPatientID))
    
     loop
      begin
      
        v_nNumOfIncAssessments := 0;
      
        --select all encounter intakes responses for module for encounter by mid
        select count(eir.encounter_id)
          into v_nNumOfIncAssessments
          from tbicds.encounter_intake_responses eir
         where eir.encounter_id = rec.encounter_id
           and eir.mid = rec.mid;
      
        if v_nNumOfIncAssessments = 0 then
        
          delete from tbicds.encounter_intake ei
           where ei.encounter_id = rec.encounter_id
             and ei.encounter_intake_id = rec.encounter_intake_id
             and ei.mid = rec.mid;
        
          commit;
        
        end if;
      
      exception
        when others then
          rollback;
      end;
    end loop;*/
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.DelIncPatIntakeAssessments(): ' ||
                           sqlErrm;
  end;

  /*
  get incomplete patient assessments
  */
  procedure IncPatIntakeAssessments(pi_vSessionID            in varchar2,
                                    pi_vSessionClientIP      in varchar2,
                                    pi_nUserID               in number,
                                    pi_vPatientID            in varchar2,
                                    po_nHasIncPatAssessments out number,
                                    po_nStatusCode           out number,
                                    po_vStatusComment        out varchar2) 
  is
  
    v_nNumOfIncAssessments number := 0;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
   /* for rec in ( -- select all INCOMPLETE encounter intakes for module for patient
                select ei.*
                  from tbicds.encounter_intake ei
                 where ei.encounter_id in
                       ( -- select all encounters for patient
                        select e.encounter_id
                          from tbicds.encounter e
                         where e.patient_id = pi_vPatientID))
    
     loop
      begin
      
        v_nNumOfIncAssessments := 0;
      
        --select all encounter intakes responses for module for encounter by mid
        select count(eir.encounter_id)
          into v_nNumOfIncAssessments
          from tbicds.encounter_intake_responses eir
         where eir.encounter_id = rec.encounter_id
           and eir.mid = rec.mid;
      
        if v_nNumOfIncAssessments = 0 then
          po_nHasIncPatAssessments := 1;
        end if;
      
      exception
        when others then
          rollback;
      end;
    end loop;
  */
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_PATIENT.IncPatIntakeAssessments(): ' ||
                           sqlErrm;
  end;


  /*
  delete patient assigned modules
  */
  procedure DeletePatientAssignedModule(pi_vPatientID     in varchar2,
                                        nMID              in number,
                                        po_nStatusCode    out number,
                                        po_vStatusComment out varchar2)
   is
    v_vSQL constant varchar2(32767) := 'delete from tbicds.patient_module where patient_id = :pi_vPatientID 
                                                                   and mid = :nMID 
                                                                   and status = 0
                                                                   and date_completed is null';
  begin
    po_nStatusCode    := PCK_COMMON.c_nResultStatus_Success;
    po_vStatusComment := '';
  
    execute immediate v_vSQL
      using pi_vPatientID, nMID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := PCK_COMMON.c_nResultStatus_Error;
      po_vStatusComment := 'PCK_PATIENT.DeletePatientAssignedModule(): ' ||
                           sqlErrm;
  end;

  /*
  Update patient user rights
  */
  procedure updatePatFxUserRights
   is
    curPat pck_common.refCursor;
    recPat tbicds.patient_demographics%rowtype;
  
  begin
  
    open curPat for
      select p.*
        from tbicds.patient_demographics p
       where p.fx_user_id is not null
         and p.fx_user_id not in
             (select fx_user_id from tbicds.fx_user_rights);
  
    loop
      fetch curPat
        into recPat;
      exit when curPat%notfound;
    
      insert into tbicds.fx_user_rights
        (fx_user_id, user_rights, read_only, user_type)
      values
        (recPat.Fx_User_Id, 0, 0, 0);
      commit;
    
    end loop;
    close curPat;
  
  end;

  /*
  Returns patient socioeconomics recordset 
  */
  procedure GetPatientSocioeconomicsRS(pi_vSessionID       in varchar2,
                                       pi_vSessionClientIP in varchar2,
                                       pi_nUserID          in number,
                                       pi_vKey             in varchar2,
                                       pi_vPatientID       in varchar2,
                                       po_nStatusCode      out number,
                                       po_vStatusComment   out varchar2,
                                       rs                  out RetRefCursor)
  is
  begin
    po_nStatusCode    := 0;
    po_vStatusComment := '';
    open rs for
      select patient_id,
             highest_grade_id,
             highest_degree_id,
             marital_status_id,
             employment_status_id,
             job_classification_id,
             last_modified,
             last_modified_by,
             tbicds.fnc_utl_decstr(t.employment_status_other,
                                   pi_vKey,
                                   t.patient_id)
        from patient_socioeconomic t
       where t.patient_id = pi_vPatientID;
  
  exception
    when others then
      po_nStatusCode    := -1;
      po_vStatusComment := 'PCK_PATIENT.GetPatientSocioeconomicsRS(): ' ||
                           sqlCode || ': ' || sqlErrm;
  end;

  /*
  Updates patient socioeconomics data 
  */
  procedure UpdatePatientSocioeconomics(pi_vSessionID             in varchar2,
                                        pi_vSessionClientIP       in varchar2,
                                        pi_nUserID                in number,
                                        pi_vKey                   in varchar2,
                                        pi_vPatientID             in varchar2,
                                        pi_nHighestGradeID        in number,
                                        pi_nHighestDegreeID       in number,
                                        pi_nMaritalStatusID       in number,
                                        pi_nEmploymentStatusID    in number,
                                        pi_nJobClassificationID   in number,
                                        pi_vEmploymentStatusOther in varchar2,
                                        po_nStatusCode            out number,
                                        po_vStatusComment         out varchar2)
  is
  
    v_nHighestGradeID        number := null;
    v_nHighestDegreeID       number := null;
    v_nMaritalStatusID       number := null;
    v_nEmploymentStatusID    number := null;
    v_nJobClassificationID   number := null;
    v_vEmploymentStatusOther varchar2(255) := null;
  
    v_nCount number := 0;
  
  begin
    po_nStatusCode    := 0;
    po_vStatusComment := '';
  
    --check if the patient record exists
    begin
      select count(*)
        into v_nCount
        from patient_socioeconomic t
       where t.patient_id = pi_vPatientID;
    exception
      when others then
        v_nCount := 0;
    end;
  
    if v_nCount < 1 then
      --insert record
      InsertPatientSocioeconomics(pi_vSessionID             => pi_vSessionID,
                                  pi_vSessionClientIP       => pi_vSessionClientIP,
                                  pi_nUserID                => pi_nUserID,
                                  pi_vKey                   => pi_vKey,
                                  pi_vPatientID             => pi_vPatientID,
                                  pi_nHighestGradeID        => pi_nHighestGradeID,
                                  pi_nHighestDegreeID       => pi_nHighestDegreeID,
                                  pi_nMaritalStatusID       => pi_nMaritalStatusID,
                                  pi_nEmploymentStatusID    => pi_nEmploymentStatusID,
                                  pi_nJobClassificationID   => pi_nJobClassificationID,
                                  pi_vEmploymentStatusOther => pi_vEmploymentStatusOther,
                                  po_nStatusCode            => po_nStatusCode,
                                  po_vStatusComment         => po_vStatusComment);
      return;
    end if;
  
    if pi_nHighestGradeID > 0 then
      v_nHighestGradeID := pi_nHighestGradeID;
    end if;
  
    if pi_nHighestDegreeID > 0 then
      v_nHighestDegreeID := pi_nHighestDegreeID;
    end if;
  
    if pi_nMaritalStatusID > 0 then
      v_nMaritalStatusID := pi_nMaritalStatusID;
    end if;
  
    if pi_nEmploymentStatusID > 0 then
      v_nEmploymentStatusID := pi_nEmploymentStatusID;
    end if;
  
    if pi_nJobClassificationID > 0 then
      v_nJobClassificationID := pi_nJobClassificationID;
    end if;
  
    if length(pi_vEmploymentStatusOther) > 0 then
      v_vEmploymentStatusOther := pi_vEmploymentStatusOther;
    end if;
  
    update patient_socioeconomic t
       set t.highest_grade_id        = v_nHighestGradeID,
           t.highest_degree_id       = v_nHighestDegreeID,
           t.marital_status_id       = v_nMaritalStatusID,
           t.employment_status_id    = v_nEmploymentStatusID,
           t.job_classification_id   = v_nJobClassificationID,
           t.employment_status_other = v_vEmploymentStatusOther,
           t.last_modified           = sysdate,
           t.last_modified_by        = pi_nUserID
     where t.patient_id = pi_vPatientID;
  
    commit;
  
  exception
    when others then
      po_nStatusCode    := -1;
      po_vStatusComment := 'PCK_PATIENT.UpdatePatientSocioeconomics(): ' ||
                           sqlCode || ': ' || sqlErrm;
  end;

  /*
  Inserts patient socioeconomics data 
  */
  procedure InsertPatientSocioeconomics(pi_vSessionID             in varchar2,
                                        pi_vSessionClientIP       in varchar2,
                                        pi_nUserID                in number,
                                        pi_vKey                   in varchar2,
                                        pi_vPatientID             in varchar2,
                                        pi_nHighestGradeID        in number,
                                        pi_nHighestDegreeID       in number,
                                        pi_nMaritalStatusID       in number,
                                        pi_nEmploymentStatusID    in number,
                                        pi_nJobClassificationID   in number,
                                        pi_vEmploymentStatusOther in varchar2,
                                        po_nStatusCode            out number,
                                        po_vStatusComment         out varchar2)
  is
  
    v_nHighestGradeID        number := null;
    v_nHighestDegreeID       number := null;
    v_nMaritalStatusID       number := null;
    v_nEmploymentStatusID    number := null;
    v_nJobClassificationID   number := null;
    v_vEmploymentStatusOther varchar2(255) := null;
  
  begin
    po_nStatusCode    := 0;
    po_vStatusComment := '';
  
    if pi_nHighestGradeID > 0 then
      v_nHighestGradeID := pi_nHighestGradeID;
    end if;
  
    if pi_nHighestDegreeID > 0 then
      v_nHighestDegreeID := pi_nHighestDegreeID;
    end if;
  
    if pi_nMaritalStatusID > 0 then
      v_nMaritalStatusID := pi_nMaritalStatusID;
    end if;
  
    if pi_nEmploymentStatusID > 0 then
      v_nEmploymentStatusID := pi_nEmploymentStatusID;
    end if;
  
    if pi_nJobClassificationID > 0 then
      v_nJobClassificationID := pi_nJobClassificationID;
    end if;
  
    if length(pi_vEmploymentStatusOther) > 0 then
      v_vEmploymentStatusOther := pi_vEmploymentStatusOther;
    end if;
  
    insert into patient_socioeconomic
      (patient_id,
       highest_grade_id,
       highest_degree_id,
       marital_status_id,
       employment_status_id,
       job_classification_id,
       last_modified,
       last_modified_by,
       employment_status_other)
    values
      (pi_vPatientID,
       v_nHighestDegreeID,
       v_nHighestDegreeID,
       v_nMaritalStatusID,
       v_nEmploymentStatusID,
       v_nJobClassificationID,
       sysdate,
       pi_nUserID,
       v_vEmploymentStatusOther);
  
    commit;
  
  exception
    when others then
      po_nStatusCode    := -1;
      po_vStatusComment := 'PCK_PATIENT.InsertPatientSocioeconomics(): ' ||
                           sqlCode || ': ' || sqlErrm;
  end;

end;
/

